<?php
/**
 * OptiCore SaaS - FormaPagoController
 */

class FormaPagoController
{
    private FormaPago $model;

    public function __construct()
    {
        Auth::requirePermission('formas_pago.ver');
        $this->model = new FormaPago();
    }

    // ── GET /formas-pago ──────────────────────────────────────
    public function index(): void
    {
        $page      = currentPage();
        $busqueda  = get('q');
        $resultado = $this->model->getAll($page, $busqueda);

        view('formas_pago.index', [
            'title'       => 'Formas de Pago',
            'formas_pago' => $resultado['data'],
            'pagination'  => $resultado,
            'busqueda'    => $busqueda,
        ]);
    }

    // ── GET /formas-pago/create ───────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('formas_pago.crear');
        view('formas_pago.create', ['title' => 'Nueva Forma de Pago']);
    }

    // ── POST /formas-pago ─────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('formas_pago.crear');
        csrf_verify();

        $errors = validateRequired($_POST, [
            'nombre' => 'Nombre',
            'tipo'   => 'Tipo',
        ]);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/formas-pago/create');
        }

        $id = $this->model->crear($_POST);
        AuditLog::log('crear_forma_pago', 'formas_pago', $id);
        flash('success', 'Forma de pago creada correctamente.');
        redirect('/formas-pago');
    }

    // ── GET /formas-pago/{id}/edit ────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('formas_pago.editar');

        $formaPago = $this->model->getById($id);
        if (!$formaPago) {
            flash('error', 'Forma de pago no encontrada.');
            redirect('/formas-pago');
        }

        view('formas_pago.edit', [
            'title'      => 'Editar Forma de Pago',
            'forma_pago' => $formaPago,
        ]);
    }

    // ── POST /formas-pago/{id} ────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('formas_pago.editar');
        csrf_verify();

        $formaPago = $this->model->getById($id);
        if (!$formaPago) {
            flash('error', 'Forma de pago no encontrada.');
            redirect('/formas-pago');
        }

        $errors = validateRequired($_POST, [
            'nombre' => 'Nombre',
            'tipo'   => 'Tipo',
        ]);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/formas-pago/$id/edit");
        }

        $this->model->actualizar($id, $_POST);
        AuditLog::log('editar_forma_pago', 'formas_pago', $id, $formaPago, $_POST);
        flash('success', 'Forma de pago actualizada correctamente.');
        redirect('/formas-pago');
    }

    // ── POST /formas-pago/{id}/delete ─────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('formas_pago.eliminar');
        csrf_verify();

        $formaPago = $this->model->getById($id);
        if (!$formaPago) {
            flash('error', 'Forma de pago no encontrada.');
            redirect('/formas-pago');
        }

        if ($this->model->tieneUso($id)) {
            flash('error', 'No se puede eliminar: la forma de pago tiene ventas asociadas.');
            redirect('/formas-pago');
        }

        $this->model->eliminar($id);
        AuditLog::log('eliminar_forma_pago', 'formas_pago', $id, $formaPago);
        flash('success', 'Forma de pago desactivada correctamente.');
        redirect('/formas-pago');
    }
}
