<?php
/**
 * OptiCore SaaS - ProductoController
 */

class ProductoController
{
    private Producto $model;

    public function __construct()
    {
        Auth::requirePermission('productos.ver');
        $this->model = new Producto();
    }

    // ── GET /productos ────────────────────────────────────────
    public function index(): void
    {
        $page      = currentPage();
        $busqueda  = get('q');
        $tipo      = get('tipo');
        $resultado = $this->model->listar($page, ITEMS_PER_PAGE, $busqueda ?: null, $tipo ?: null);

        view('productos.index', [
            'title'      => 'Productos',
            'productos'  => $resultado['data'],
            'pagination' => $resultado['pagination'],
            'busqueda'   => $busqueda,
            'tipo'       => $tipo,
        ]);
    }

    // ── GET /productos/buscar (AJAX) ──────────────────────────
    public function buscar(): void
    {
        $q         = get('q');
        $productos = $this->model->buscarParaVenta($q);
        jsonResponse($productos);
    }

    // ── GET /productos/create ─────────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('productos.crear');

        $empresaId = Auth::empresaId();
        if ($empresaId && !(new Plan())->verificarLimite($empresaId, 'productos')) {
            flash('error', 'Has alcanzado el límite de productos de tu plan.');
            redirect('/productos');
        }

        $categorias  = $this->getCategorias();
        $marcas      = $this->getMarcas();
        $proveedores = $this->getProveedores();

        view('productos.create', [
            'title'       => 'Nuevo Producto',
            'categorias'  => $categorias,
            'marcas'      => $marcas,
            'proveedores' => $proveedores,
        ]);
    }

    // ── POST /productos ───────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('productos.crear');
        csrf_verify();

        $errors = validateRequired($_POST, [
            'nombre'       => 'Nombre',
            'precio_venta' => 'Precio de venta',
            'tipo'         => 'Tipo',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/productos/create');
        }

        // Subir imagen si existe
        $imagen = null;
        if (!empty($_FILES['imagen']['name'])) {
            $imagen = uploadFile($_FILES['imagen'], 'productos', ['jpg','jpeg','png','gif','webp']);
        }

        $id = $this->model->crear(array_merge($_POST, ['imagen' => $imagen]));
        AuditLog::log('crear_producto', 'productos', $id);
        flash('success', 'Producto creado correctamente.');
        redirect('/productos');
    }

    // ── GET /productos/{id}/edit ──────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('productos.editar');

        $producto = $this->model->findById($id);
        if (!$producto) {
            flash('error', 'Producto no encontrado.');
            redirect('/productos');
        }

        $categorias  = $this->getCategorias();
        $marcas      = $this->getMarcas();
        $proveedores = $this->getProveedores();

        view('productos.edit', [
            'title'       => 'Editar Producto',
            'producto'    => $producto,
            'categorias'  => $categorias,
            'marcas'      => $marcas,
            'proveedores' => $proveedores,
        ]);
    }

    // ── POST /productos/{id} ──────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('productos.editar');
        csrf_verify();

        $producto = $this->model->findById($id);
        if (!$producto) {
            flash('error', 'Producto no encontrado.');
            redirect('/productos');
        }

        $errors = validateRequired($_POST, [
            'nombre'       => 'Nombre',
            'precio_venta' => 'Precio de venta',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/productos/$id/edit");
        }

        // Subir imagen si existe
        $imagen = $producto['imagen'];
        if (!empty($_FILES['imagen']['name'])) {
            $uploaded = uploadFile($_FILES['imagen'], 'productos', ['jpg','jpeg','png','gif','webp']);
            if ($uploaded) $imagen = $uploaded;
        }

        $this->model->actualizar($id, array_merge($_POST, ['imagen' => $imagen]));
        AuditLog::log('editar_producto', 'productos', $id, $producto, $_POST);
        flash('success', 'Producto actualizado correctamente.');
        redirect('/productos');
    }

    // ── POST /productos/{id}/delete ───────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('productos.eliminar');
        csrf_verify();

        $producto = $this->model->findById($id);
        if (!$producto) {
            flash('error', 'Producto no encontrado.');
            redirect('/productos');
        }

        $this->model->update($id, ['estado' => 'inactivo']);
        AuditLog::log('desactivar_producto', 'productos', $id, $producto);
        flash('success', 'Producto desactivado correctamente.');
        redirect('/productos');
    }

    // ── Helpers privados ──────────────────────────────────────
    private function getCategorias(): array
    {
        $empresaId = Auth::empresaId();
        if (!$empresaId) return [];
        return db()->fetchAll(
            "SELECT id, nombre FROM categorias WHERE empresa_id = ? AND estado = 'activa' ORDER BY nombre",
            [$empresaId]
        );
    }

    private function getMarcas(): array
    {
        $empresaId = Auth::empresaId();
        if (!$empresaId) return [];
        return db()->fetchAll(
            "SELECT id, nombre FROM marcas WHERE empresa_id = ? AND estado = 'activa' ORDER BY nombre",
            [$empresaId]
        );
    }

    private function getProveedores(): array
    {
        $empresaId = Auth::empresaId();
        if (!$empresaId) return [];
        return db()->fetchAll(
            "SELECT id, nombre FROM proveedores WHERE empresa_id = ? AND estado = 'activo' ORDER BY nombre",
            [$empresaId]
        );
    }
}
