<?php
/**
 * OptiCore SaaS - ProveedorController
 */

class ProveedorController
{
    private Proveedor $model;

    public function __construct()
    {
        Auth::requirePermission('proveedores.ver');
        $this->model = new Proveedor();
    }

    // ── GET /proveedores ──────────────────────────────────────
    public function index(): void
    {
        $page      = currentPage();
        $busqueda  = get('q');
        $tipo      = get('tipo');
        $resultado = $this->model->getAll($page, $busqueda, $tipo);

        view('proveedores.index', [
            'title'       => 'Proveedores',
            'proveedores' => $resultado['data'],
            'pagination'  => $resultado,
            'busqueda'    => $busqueda,
            'tipo'        => $tipo,
        ]);
    }

    // ── GET /proveedores/create ───────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('proveedores.crear');
        view('proveedores.create', ['title' => 'Nuevo Proveedor']);
    }

    // ── POST /proveedores ─────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('proveedores.crear');
        csrf_verify();

        $errors = validateRequired($_POST, ['nombre' => 'Nombre']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/proveedores/create');
        }

        if (!empty($_POST['email']) && !filter_var($_POST['email'], FILTER_VALIDATE_EMAIL)) {
            flash('error', 'El email no es válido.');
            redirect('/proveedores/create');
        }

        $id = $this->model->crear($_POST);
        AuditLog::log('crear_proveedor', 'proveedores', $id);
        flash('success', 'Proveedor creado correctamente.');
        redirect('/proveedores');
    }

    // ── GET /proveedores/{id}/edit ────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('proveedores.editar');

        $proveedor = $this->model->getById($id);
        if (!$proveedor) {
            flash('error', 'Proveedor no encontrado.');
            redirect('/proveedores');
        }

        view('proveedores.edit', [
            'title'     => 'Editar Proveedor',
            'proveedor' => $proveedor,
        ]);
    }

    // ── POST /proveedores/{id} ────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('proveedores.editar');
        csrf_verify();

        $proveedor = $this->model->getById($id);
        if (!$proveedor) {
            flash('error', 'Proveedor no encontrado.');
            redirect('/proveedores');
        }

        $errors = validateRequired($_POST, ['nombre' => 'Nombre']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/proveedores/$id/edit");
        }

        if (!empty($_POST['email']) && !filter_var($_POST['email'], FILTER_VALIDATE_EMAIL)) {
            flash('error', 'El email no es válido.');
            redirect("/proveedores/$id/edit");
        }

        $this->model->actualizar($id, $_POST);
        AuditLog::log('editar_proveedor', 'proveedores', $id, $proveedor, $_POST);
        flash('success', 'Proveedor actualizado correctamente.');
        redirect('/proveedores');
    }

    // ── POST /proveedores/{id}/delete ─────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('proveedores.eliminar');
        csrf_verify();

        $proveedor = $this->model->getById($id);
        if (!$proveedor) {
            flash('error', 'Proveedor no encontrado.');
            redirect('/proveedores');
        }

        $this->model->eliminar($id);
        AuditLog::log('eliminar_proveedor', 'proveedores', $id, $proveedor);
        flash('success', 'Proveedor desactivado correctamente.');
        redirect('/proveedores');
    }
}
