<?php
/**
 * OptiCore SaaS - SucursalController
 */

class SucursalController
{
    private Sucursal $model;

    public function __construct()
    {
        Auth::requirePermission('sucursales.ver');
        $this->model = new Sucursal();
    }

    // ── GET /sucursales ───────────────────────────────────────
    public function index(): void
    {
        $page   = currentPage();
        $result = $this->model->getAll($page);
        view('sucursales.index', [
            'title'      => 'Sucursales',
            'sucursales' => $result['data'],
            'pagination' => $result,
        ]);
    }

    // ── GET /sucursales/create ────────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('sucursales.crear');

        // Verificar límite del plan (solo para usuarios de empresa)
        $empresaId = Auth::empresaId();
        if ($empresaId && !(new Plan())->verificarLimite($empresaId, 'sucursales')) {
            flash('error', 'Has alcanzado el límite de sucursales de tu plan.');
            redirect('/sucursales');
        }

        // SuperAdmin necesita seleccionar la empresa destino
        $empresas = [];
        if (Auth::isSuperAdmin()) {
            $empresas = db()->fetchAll(
                "SELECT id, nombre FROM empresas WHERE estado = 'activa' ORDER BY nombre ASC"
            );
        }

        view('sucursales.create', [
            'title'    => 'Nueva Sucursal',
            'empresas' => $empresas,
        ]);
    }

    // ── POST /sucursales ──────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('sucursales.crear');
        csrf_verify();

        $required = ['nombre' => 'Nombre'];

        // SuperAdmin debe seleccionar empresa
        if (Auth::isSuperAdmin()) {
            if (empty($_POST['empresa_id']) || (int)$_POST['empresa_id'] <= 0) {
                flash('error', 'Debes seleccionar una empresa para la sucursal.');
                redirect('/sucursales/create');
            }
        }

        $errors = validateRequired($_POST, $required);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/sucursales/create');
        }

        $id = $this->model->crear($_POST);
        AuditLog::log('crear_sucursal', 'sucursales', $id, null, $_POST);
        flash('success', 'Sucursal creada correctamente.');
        redirect('/sucursales');
    }

    // ── GET /sucursales/{id}/edit ─────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('sucursales.editar');

        $sucursal = $this->model->findById($id);
        if (!$sucursal) {
            flash('error', 'Sucursal no encontrada.');
            redirect('/sucursales');
        }
        view('sucursales.edit', ['title' => 'Editar Sucursal', 'sucursal' => $sucursal]);
    }

    // ── POST /sucursales/{id} ─────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('sucursales.editar');
        csrf_verify();

        $sucursal = $this->model->findById($id);
        if (!$sucursal) {
            flash('error', 'Sucursal no encontrada.');
            redirect('/sucursales');
        }

        $errors = validateRequired($_POST, ['nombre' => 'Nombre']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/sucursales/$id/edit");
        }

        $this->model->update($id, [
            'nombre'    => $_POST['nombre'],
            'codigo'    => $_POST['codigo'] ?? null,
            'direccion' => $_POST['direccion'] ?? null,
            'telefono'  => $_POST['telefono'] ?? null,
            'email'     => $_POST['email'] ?? null,
            'horario'   => $_POST['horario'] ?? null,
            'estado'    => $_POST['estado'] ?? 'activa',
        ]);

        AuditLog::log('editar_sucursal', 'sucursales', $id, $sucursal, $_POST);
        flash('success', 'Sucursal actualizada correctamente.');
        redirect('/sucursales');
    }

    // ── POST /sucursales/{id}/delete ──────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('sucursales.eliminar');
        csrf_verify();

        $sucursal = $this->model->findById($id);
        if (!$sucursal) {
            flash('error', 'Sucursal no encontrada.');
            redirect('/sucursales');
        }

        if ($sucursal['es_principal']) {
            flash('error', 'No se puede eliminar la sucursal principal.');
            redirect('/sucursales');
        }

        $this->model->delete($id);
        AuditLog::log('eliminar_sucursal', 'sucursales', $id, $sucursal);
        flash('success', 'Sucursal eliminada correctamente.');
        redirect('/sucursales');
    }
}
