<?php
/**
 * OptiCore SaaS - Auth Middleware
 * Verifica autenticación y estado de licencia en cada request
 */

class AuthMiddleware
{
    public static function handle(): void
    {
        // 1. Verificar autenticación
        if (!Auth::check()) {
            if (isAjax()) {
                jsonResponse(['error' => 'No autenticado', 'redirect' => BASE_URL . '/login'], 401);
            }
            flash('error', 'Debes iniciar sesión para continuar.');
            redirect('/login');
        }

        $user = Auth::user();

        // 2. SuperAdmin: acceso total sin restricciones de licencia
        if (Auth::isSuperAdmin()) {
            return;
        }

        // 3. Verificar que la empresa esté activa
        if (empty($user['empresa_id'])) {
            Auth::logout();
            flash('error', 'Tu cuenta no tiene empresa asignada. Contacta al administrador.');
            redirect('/login');
        }

        // 4. Verificar licencia activa (solo si hay empresa)
        self::verificarLicencia((int) $user['empresa_id']);
    }

    // ── Verificar licencia de la empresa ──────────────────────
    private static function verificarLicencia(int $empresaId): void
    {
        // Cache en sesión para no consultar DB en cada request
        $cacheKey  = 'licencia_check_' . $empresaId;
        $cacheTime = $_SESSION[$cacheKey . '_time'] ?? 0;

        // Refrescar cada 5 minutos
        if (time() - $cacheTime < 300 && isset($_SESSION[$cacheKey])) {
            if (!$_SESSION[$cacheKey]) {
                self::licenciaExpirada();
            }
            return;
        }

        $db      = db();
        $empresa = $db->fetchOne(
            "SELECT e.estado, l.estado as licencia_estado, l.fecha_fin
             FROM empresas e
             LEFT JOIN licencias l ON l.empresa_id = e.id AND l.estado = 'activa'
             WHERE e.id = ?
             ORDER BY l.fecha_fin DESC
             LIMIT 1",
            [$empresaId]
        );

        $valida = false;
        if ($empresa) {
            $estadoEmpresa = $empresa['estado'];
            if ($estadoEmpresa === 'activa') {
                // Verificar licencia
                if ($empresa['licencia_estado'] === 'activa') {
                    // Verificar fecha de vencimiento
                    if (empty($empresa['fecha_fin']) || $empresa['fecha_fin'] >= date('Y-m-d')) {
                        $valida = true;
                    }
                }
            }
        }

        // Guardar en caché de sesión
        $_SESSION[$cacheKey]            = $valida;
        $_SESSION[$cacheKey . '_time']  = time();

        if (!$valida) {
            self::licenciaExpirada();
        }
    }

    // ── Manejar licencia expirada ─────────────────────────────
    private static function licenciaExpirada(): never
    {
        Auth::logout();
        if (isAjax()) {
            jsonResponse([
                'error'    => 'Licencia expirada o suspendida.',
                'redirect' => BASE_URL . '/login',
            ], 403);
        }
        flash('error', 'Tu licencia ha expirado o está suspendida. Contacta al administrador.');
        redirect('/login');
    }

    // ── Limpiar caché de licencia ─────────────────────────────
    public static function clearLicenciaCache(int $empresaId): void
    {
        $cacheKey = 'licencia_check_' . $empresaId;
        unset($_SESSION[$cacheKey], $_SESSION[$cacheKey . '_time']);
    }
}
