<?php
/**
 * OptiCore SaaS - Base Model con scoping multi-tenant
 */

abstract class BaseModel
{
    protected Database $db;
    protected string $table = '';
    protected ?int $empresaId;

    public function __construct()
    {
        $this->db        = db();
        $this->empresaId = Auth::empresaId();
    }

    // ── Scope empresa (multi-tenant) ──────────────────────────
    protected function empresaScope(): int
    {
        if (!$this->empresaId && !Auth::isSuperAdmin()) {
            throw new RuntimeException('empresa_id requerido para operaciones multi-tenant');
        }
        return $this->empresaId ?? 0;
    }

    // ── Buscar por ID con scope empresa ───────────────────────
    public function findById(int $id): array|false
    {
        if (Auth::isSuperAdmin()) {
            return $this->db->fetchOne("SELECT * FROM `{$this->table}` WHERE id = ?", [$id]);
        }
        return $this->db->fetchOne(
            "SELECT * FROM `{$this->table}` WHERE id = ? AND empresa_id = ?",
            [$id, $this->empresaScope()]
        );
    }

    // ── Todos los registros de la empresa ─────────────────────
    public function all(string $orderBy = 'id DESC'): array
    {
        if (Auth::isSuperAdmin()) {
            return $this->db->fetchAll("SELECT * FROM `{$this->table}` ORDER BY $orderBy");
        }
        return $this->db->fetchAll(
            "SELECT * FROM `{$this->table}` WHERE empresa_id = ? ORDER BY $orderBy",
            [$this->empresaScope()]
        );
    }

    // ── Insertar con empresa_id automático ────────────────────
    public function create(array $data): int
    {
        if (!isset($data['empresa_id'])) {
            $data['empresa_id'] = $this->empresaScope();
        }
        return $this->db->insert($this->table, $data);
    }

    // ── Actualizar con verificación de empresa ────────────────
    public function update(int $id, array $data): int
    {
        $where = ['id' => $id];
        if (!Auth::isSuperAdmin()) {
            $where['empresa_id'] = $this->empresaScope();
        }
        return $this->db->update($this->table, $data, $where);
    }

    // ── Eliminar con verificación de empresa ──────────────────
    public function delete(int $id): int
    {
        $where = ['id' => $id];
        if (!Auth::isSuperAdmin()) {
            $where['empresa_id'] = $this->empresaScope();
        }
        return $this->db->delete($this->table, $where);
    }

    // ── Contar registros de la empresa ────────────────────────
    public function countAll(): int
    {
        if (Auth::isSuperAdmin()) {
            return $this->db->count($this->table);
        }
        return $this->db->count($this->table, ['empresa_id' => $this->empresaScope()]);
    }
}
