<?php
/**
 * OptiCore SaaS - Plan Model
 * Gestión de planes de suscripción (solo SuperAdmin)
 */

class Plan extends BaseModel
{
    protected string $table = 'planes';

    // ── Listar todos los planes ───────────────────────────────
    public function getAll(): array
    {
        return $this->db->fetchAll(
            "SELECT p.*, COUNT(e.id) as total_empresas
             FROM planes p
             LEFT JOIN empresas e ON e.plan_id = p.id
             GROUP BY p.id
             ORDER BY p.precio_mensual ASC"
        );
    }

    // ── Obtener por ID ────────────────────────────────────────
    public function getById(int $id): array|false
    {
        return $this->db->fetchOne("SELECT * FROM planes WHERE id = ?", [$id]);
    }

    // ── Planes activos (para selects) ─────────────────────────
    public function getActivos(): array
    {
        return $this->db->fetchAll(
            "SELECT id, nombre, precio_mensual, precio_anual,
                    max_sucursales, max_usuarios, max_productos, max_pacientes
             FROM planes WHERE estado = 'activo' ORDER BY precio_mensual ASC"
        );
    }

    // ── Crear plan ────────────────────────────────────────────
    public function crear(array $data): int
    {
        return $this->db->insert('planes', [
            'nombre'            => $data['nombre'],
            'descripcion'       => $data['descripcion'] ?? null,
            'precio_mensual'    => $data['precio_mensual'] ?? 0,
            'precio_anual'      => $data['precio_anual'] ?? 0,
            'max_sucursales'    => $data['max_sucursales'] ?? 1,
            'max_usuarios'      => $data['max_usuarios'] ?? 5,
            'max_productos'     => $data['max_productos'] ?? 100,
            'max_pacientes'     => $data['max_pacientes'] ?? 500,
            'tiene_laboratorio' => isset($data['tiene_laboratorio']) ? 1 : 0,
            'tiene_agenda'      => isset($data['tiene_agenda']) ? 1 : 0,
            'tiene_reportes'    => isset($data['tiene_reportes']) ? 1 : 0,
            'tiene_api'         => isset($data['tiene_api']) ? 1 : 0,
            'estado'            => $data['estado'] ?? 'activo',
        ]);
    }

    // ── Actualizar plan ───────────────────────────────────────
    public function actualizar(int $id, array $data): int
    {
        return $this->db->update('planes', [
            'nombre'            => $data['nombre'],
            'descripcion'       => $data['descripcion'] ?? null,
            'precio_mensual'    => $data['precio_mensual'] ?? 0,
            'precio_anual'      => $data['precio_anual'] ?? 0,
            'max_sucursales'    => $data['max_sucursales'] ?? 1,
            'max_usuarios'      => $data['max_usuarios'] ?? 5,
            'max_productos'     => $data['max_productos'] ?? 100,
            'max_pacientes'     => $data['max_pacientes'] ?? 500,
            'tiene_laboratorio' => isset($data['tiene_laboratorio']) ? 1 : 0,
            'tiene_agenda'      => isset($data['tiene_agenda']) ? 1 : 0,
            'tiene_reportes'    => isset($data['tiene_reportes']) ? 1 : 0,
            'tiene_api'         => isset($data['tiene_api']) ? 1 : 0,
            'estado'            => $data['estado'] ?? 'activo',
        ], ['id' => $id]);
    }

    // ── Eliminar plan (solo si no tiene empresas) ─────────────
    public function eliminar(int $id): bool
    {
        $count = $this->db->count('empresas', ['plan_id' => $id]);
        if ($count > 0) return false;
        $this->db->delete('planes', ['id' => $id]);
        return true;
    }

    // ── Verificar límites del plan para una empresa ───────────
    public function verificarLimite(int $empresaId, string $limite): bool
    {
        $plan = $this->db->fetchOne(
            "SELECT p.* FROM planes p
             JOIN empresas e ON e.plan_id = p.id
             WHERE e.id = ?",
            [$empresaId]
        );
        if (!$plan) return false;

        $campo = 'max_' . $limite;
        if (!isset($plan[$campo])) return true;

        $tabla = match($limite) {
            'sucursales' => 'sucursales',
            'usuarios'   => 'usuarios',
            'productos'  => 'productos',
            'pacientes'  => 'pacientes',
            default      => null,
        };
        if (!$tabla) return true;

        $actual = $this->db->count($tabla, ['empresa_id' => $empresaId]);
        return $actual < (int) $plan[$campo];
    }
}
