<?php
/**
 * OptiCore SaaS - Receta Model
 */

class Receta extends BaseModel
{
    protected string $table = 'recetas';

    public function getAll(int $page = 1, string $search = '', ?int $sucursalId = null): array
    {
        $params = [$this->empresaScope()];
        $where  = 'r.empresa_id = ?';

        if ($search) {
            $where   .= " AND (CONCAT(p.nombre,' ',p.apellido) LIKE ? OR p.rut LIKE ?)";
            $s        = "%$search%";
            $params[] = $s; $params[] = $s;
        }
        if ($sucursalId) {
            $where   .= ' AND r.sucursal_id = ?';
            $params[] = $sucursalId;
        }

        $sql = "SELECT r.*,
                       CONCAT(p.nombre,' ',p.apellido) as paciente_nombre,
                       p.rut as paciente_rut,
                       CONCAT(u.nombre,' ',u.apellido) as profesional,
                       s.nombre as sucursal_nombre
                FROM recetas r
                LEFT JOIN pacientes p ON r.paciente_id = p.id
                LEFT JOIN usuarios u ON r.usuario_id = u.id
                LEFT JOIN sucursales s ON r.sucursal_id = s.id
                WHERE $where
                ORDER BY r.fecha DESC, r.id DESC";

        return $this->db->paginate($sql, $params, $page);
    }

    public function getById(int $id): array|false
    {
        return $this->db->fetchOne(
            "SELECT r.*,
                    CONCAT(p.nombre,' ',p.apellido) as paciente_nombre,
                    p.rut as paciente_rut, p.telefono as paciente_telefono,
                    p.fecha_nacimiento as paciente_nacimiento,
                    CONCAT(u.nombre,' ',u.apellido) as profesional,
                    s.nombre as sucursal_nombre, s.direccion as sucursal_direccion,
                    e.nombre as empresa_nombre, e.direccion as empresa_direccion,
                    e.telefono as empresa_telefono, e.logo as empresa_logo
             FROM recetas r
             LEFT JOIN pacientes p ON r.paciente_id = p.id
             LEFT JOIN usuarios u ON r.usuario_id = u.id
             LEFT JOIN sucursales s ON r.sucursal_id = s.id
             LEFT JOIN empresas e ON r.empresa_id = e.id
             WHERE r.id = ? AND r.empresa_id = ?",
            [$id, $this->empresaScope()]
        );
    }

    public function getByPaciente(int $pacienteId): array
    {
        return $this->db->fetchAll(
            "SELECT r.*, CONCAT(u.nombre,' ',u.apellido) as profesional
             FROM recetas r
             LEFT JOIN usuarios u ON r.usuario_id = u.id
             WHERE r.paciente_id = ? AND r.empresa_id = ?
             ORDER BY r.fecha DESC",
            [$pacienteId, $this->empresaScope()]
        );
    }

    public function getUltimaByPaciente(int $pacienteId): array|false
    {
        return $this->db->fetchOne(
            "SELECT * FROM recetas WHERE paciente_id = ? AND empresa_id = ? AND estado = 'activa'
             ORDER BY fecha DESC LIMIT 1",
            [$pacienteId, $this->empresaScope()]
        );
    }

    public function countByPeriodo(string $desde, string $hasta): int
    {
        return (int) $this->db->fetchColumn(
            "SELECT COUNT(*) FROM recetas WHERE empresa_id = ? AND fecha BETWEEN ? AND ?",
            [$this->empresaScope(), $desde, $hasta]
        );
    }

    // ── Listar con paginación (usado por el controlador) ──────
    public function listar(int $page = 1, int $perPage = ITEMS_PER_PAGE, ?int $pacienteId = null): array
    {
        $params = [$this->empresaScope()];
        $where  = 'r.empresa_id = ?';

        if ($pacienteId) {
            $where   .= ' AND r.paciente_id = ?';
            $params[] = $pacienteId;
        }

        $sql = "SELECT r.*,
                       CONCAT(p.nombre,' ',p.apellido) as paciente_nombre,
                       p.rut as paciente_rut,
                       CONCAT(u.nombre,' ',u.apellido) as profesional,
                       s.nombre as sucursal_nombre
                FROM recetas r
                LEFT JOIN pacientes p ON r.paciente_id = p.id
                LEFT JOIN usuarios u ON r.usuario_id = u.id
                LEFT JOIN sucursales s ON r.sucursal_id = s.id
                WHERE $where
                ORDER BY r.fecha DESC, r.id DESC";

        $result = $this->db->paginate($sql, $params, $page, $perPage);
        return [
            'data'       => $result['data'],
            'pagination' => $result,
        ];
    }

    // ── Crear receta ──────────────────────────────────────────
    public function crear(array $data): int
    {
        return $this->db->insert('recetas', [
            'empresa_id'    => $this->empresaScope(),
            'sucursal_id'   => Auth::sucursalId() ?: null,
            'paciente_id'   => $data['paciente_id'],
            'usuario_id'    => $data['usuario_id'],
            'fecha'         => $data['fecha'],
            'tipo'          => $data['tipo'],
            'od_esfera'     => ($data['od_esfera'] ?? '') !== '' ? $data['od_esfera'] : null,
            'od_cilindro'   => ($data['od_cilindro'] ?? '') !== '' ? $data['od_cilindro'] : null,
            'od_eje'        => ($data['od_eje'] ?? '') !== '' ? $data['od_eje'] : null,
            'od_adicion'    => ($data['od_adicion'] ?? '') !== '' ? $data['od_adicion'] : null,
            'od_av'         => $data['od_av'] ?? null,
            'oi_esfera'     => ($data['oi_esfera'] ?? '') !== '' ? $data['oi_esfera'] : null,
            'oi_cilindro'   => ($data['oi_cilindro'] ?? '') !== '' ? $data['oi_cilindro'] : null,
            'oi_eje'        => ($data['oi_eje'] ?? '') !== '' ? $data['oi_eje'] : null,
            'oi_adicion'    => ($data['oi_adicion'] ?? '') !== '' ? $data['oi_adicion'] : null,
            'oi_av'         => $data['oi_av'] ?? null,
            'dp'            => ($data['dp'] ?? '') !== '' ? $data['dp'] : null,
            'observaciones' => trim($data['observaciones'] ?? '') ?: null,
            'adjunto'       => $data['adjunto'] ?? null,
            'estado'        => 'activa',
        ]);
    }

    // ── Anular receta ─────────────────────────────────────────
    public function anular(int $id): int
    {
        return $this->db->update(
            'recetas',
            ['estado' => 'anulada'],
            ['id' => $id, 'empresa_id' => $this->empresaScope()]
        );
    }
}
